<?php
/*
Plugin Name: Photomentor - Photo Elements for Elementor
Plugin URI: http://pixel-mafia.com
Description: Pixel-Mafia Premium Photo Elements for Elementor Plugin.
Version: 6.0
Author: Pixel-Mafia
Author URI: http://pixel-mafia.com
Text Domain: photomentor
*/

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Main Photomentor Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Photomentor_Plugin
{
	/**
	 * Photomentor Version
	 *
	 * Holds the version of the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const  VERSION = '6.0';
	/**
	 * Minimum Elementor Version
	 *
	 * Holds the minimum Elementor version required to run the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const  MINIMUM_ELEMENTOR_VERSION = '2.0.0';
	/**
	 * Minimum PHP Version
	 *
	 * Holds the minimum PHP version required to run the plugin.
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const  MINIMUM_PHP_VERSION = '5.4';
	/**
	 * Instance
	 *
	 * Holds a single instance of the `Photomentor_Plugin` class.
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Photomentor_Plugin A single instance of the class.
	 */
	private static  $_instance = null ;
	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Photomentor_Plugin An instance of the class.
	 */
	public static function instance() {
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 *
	 * Initialize the Photomentor plugin.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action('init', [$this, 'i18n']);
		add_action('plugins_loaded', [$this, 'init']);
	}

	/**
	 * Load Text Domain
	 *
	 * Load plugin localization files.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function i18n() {
		load_plugin_textdomain('photomentor');
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {
		// Check if Elementor installed and activated
		if ( !did_action('elementor/loaded')) {
			add_action('admin_notices', [$this, 'photomentor_admin_notice_missing_main_plugin']);
			return;
		}

		// Check for required Elementor version
		if ( !version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
			add_action('admin_notices', [$this, 'photomentor_admin_notice_minimum_elementor_version']);
			return;
		}

		// Check for required PHP version
		if ( version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' )) {
			add_action('admin_notices', [$this, 'photomentor_admin_notice_minimum_php_version']);
			return;
		}

		// Include Additional Files
		add_action('elementor/init', [$this, 'photomentor_include_additional_files']);

		// Add new Elementor Categories
		add_action('elementor/init', [$this, 'photomentor_add_elementor_category']);

		// Register Widget Scripts
		add_action('elementor/frontend/after_register_scripts', [$this, 'photomentor_register_widget_scripts']);

		add_action('wp_enqueue_scripts', function () {
			wp_localize_script('ajax_query_posts', 'photomentor_ajaxurl',
				array(
					'url' => admin_url('admin-ajax.php')
				)
			);

			wp_localize_script('ajax_query_products', 'photomentor_ajaxurl',
				array(
					'url' => admin_url('admin-ajax.php')
				)
			);

			wp_localize_script('gallery_grid', 'photomentor_ajaxurl',
				array(
					'url' => admin_url('admin-ajax.php')
				)
			);

			wp_localize_script('gallery_scattered', 'photomentor_ajaxurl',
				array(
					'url' => admin_url('admin-ajax.php')
				)
			);
		});

		// Main JS
		add_action('wp_enqueue_scripts', function (){
			wp_enqueue_script('photomentor', plugins_url('photomentor/js/lib/photomentor.js'), array('jquery'), false, true);
		});

		// Register Widget Styles
		add_action('elementor/frontend/after_enqueue_styles', [$this, 'photomentor_register_widget_styles']);

		// Register New Widgets
		add_action('elementor/widgets/widgets_registered', [$this, 'photomentor_widgets_register']);

        // Register Editor Styles
        add_action( 'elementor/editor/before_enqueue_scripts', function() {
            wp_enqueue_style('photomentor_editor', plugins_url('photomentor/css/editor.css'));
        });
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 */
	public function photomentor_admin_notice_missing_main_plugin() {
		$message = sprintf(
		/* translators: 1: Photomentor 2: Elementor */
			esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'photomentor'),
			'<strong>' . esc_html__('Photomentor', 'photomentor') . '</strong>',
			'<strong>' . esc_html__('Elementor', 'photomentor') . '</strong>'
		);
		printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.1.0
	 *
	 * @access public
	 */
	public function photomentor_admin_notice_minimum_elementor_version() {
		$message = sprintf(
		/* translators: 1: Photomentor 2: Elementor 3: Required Elementor version */
			esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'photomentor'),
			'<strong>' . esc_html__('Photomentor', 'photomentor') . '</strong>',
			'<strong>' . esc_html__('Elementor', 'photomentor') . '</strong>',
			self::MINIMUM_ELEMENTOR_VERSION
		);
		printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function photomentor_admin_notice_minimum_php_version() {
		$message = sprintf(
		/* translators: 1: Press Elements 2: PHP 3: Required PHP version */
			esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'photomentor'),
			'<strong>' . esc_html__('Press Elements', 'photomentor') . '</strong>',
			'<strong>' . esc_html__('PHP', 'photomentor') . '</strong>',
			self::MINIMUM_PHP_VERSION
		);
		printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
	}

	/**
	 * Include Additional Files
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function photomentor_include_additional_files() {
		require_once __DIR__ . '/core/ajax.php';
		require_once __DIR__ . '/core/functions.php';
		require_once __DIR__ . '/core/aq_resizer.php';
	}

	/**
	 * Add new Elementor Categories
	 *
	 * Register new widget categories for Photomentor widgets.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function photomentor_add_elementor_category() {
		\Elementor\Plugin::$instance->elements_manager->add_category(
			'photomentor_gallery_widgets',
			[
				'title' => esc_html__('Photomentor Widgets', 'photomentor'),
				'icon' => 'pm-icon-plug', //default icon
			],
			5 // position
		);

	}

	/**
	 * Register Widget Scripts
	 *
	 * Register custom scripts required to run Photomentor Widgets.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
    public function photomentor_register_widget_scripts() {
		wp_register_script('isotope', plugins_url('photomentor/js/lib/isotope.pkgd.min.js'), array('jquery', 'imagesloaded'));
		wp_register_script('event_swipe', plugins_url('photomentor/js/lib/jquery.event.swipe.js'), array('jquery'));
		wp_register_script('mouse_wheel', plugins_url('photomentor/js/lib/jquery.mousewheel.js'), array('jquery'));
		wp_register_script('owl_carousel', plugins_url('photomentor/js/lib/owl.carousel.min.js'), array('jquery'));
		wp_register_script('photoswipe', plugins_url('photomentor/js/lib/photoswipe.min.js'), array('jquery'));
		wp_register_script('photoswipe_ui', plugins_url('photomentor/js/lib/photoswipe-ui-default.min.js'), array('jquery'));
		wp_register_script('isotope_packery', plugins_url('photomentor/js/lib/packery-mode.pkgd.js'), array('jquery'));
		wp_register_script('justified_gallery', plugins_url('photomentor/js/lib/jquery.justifiedGallery.min.js'), array('jquery'));

		wp_register_script('gallery_grid', plugins_url('photomentor/js/gallery-grid-widget.js'), array('jquery', 'photoswipe', 'photoswipe_ui', 'isotope'));
		wp_register_script('gallery_grid_filter', plugins_url('photomentor/js/gallery-grid-filter-widget.js'), array('jquery', 'photoswipe', 'photoswipe_ui', 'isotope'));
		wp_register_script('gallery_justified', plugins_url('photomentor/js/gallery-justified-widget.js'), array('jquery', 'photoswipe', 'photoswipe_ui', 'justified_gallery'));
		wp_register_script('gallery_kenburns', plugins_url('photomentor/js/gallery-ken-burns-widget.js'), array('jquery'));
		wp_register_script('gallery_packery', plugins_url('photomentor/js/gallery-packery-widget.js'), array('jquery', 'photoswipe', 'photoswipe_ui', 'isotope', 'isotope_packery'));
		wp_register_script('gallery_ribbon', plugins_url('photomentor/js/gallery-ribbon-widget.js'), array('jquery', 'owl_carousel'));
		wp_register_script('gallery_slider', plugins_url('photomentor/js/gallery-slider-widget.js'), array('jquery'));
		wp_register_script('gallery_split', plugins_url('photomentor/js/gallery-split-widget.js'), array('jquery', 'event_swipe', 'mouse_wheel'));
		wp_register_script('before-after', plugins_url('photomentor/js/before-after.js'), array('jquery'));
	}

	/**
	 * Register Widget Styles
	 *
	 * Register custom styles required to run Photomentor Widgets.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function photomentor_register_widget_styles() {
		// Photomentor Plugin General Styles
		wp_register_style('photomentor', plugins_url('photomentor/css/photomentor.css'));
		wp_register_style('photomentor-icons', plugins_url('photomentor/css/photomentor-icons.css'));
		wp_enqueue_style('photomentor');
		wp_enqueue_style('photomentor-icons');

		// OWL Carousel
		wp_register_style('owl_carousel', plugins_url('photomentor/css/lib/owl.carousel.css'));
		wp_enqueue_style('owl_carousel');

		// Photoswipe
		wp_register_style('photomentor_photoswipe', plugins_url('photomentor/css/lib/photoswipe.css'));
		wp_enqueue_style('photomentor_photoswipe');

		// Justified Gallery
		wp_register_style('justified_gallery', plugins_url('photomentor/css/lib/justifiedGallery.css'));
		wp_enqueue_style('justified_gallery');
	}

	/**
	 * Register New Widgets
	 *
	 * Include Press Elements widgets files and register them in Elementor.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function photomentor_widgets_register() {
		// ---------------------------- //
		// --- Include Widget files --- //
		// ---------------------------- //
		require_once __DIR__ . '/core/widgets/gallery-grid.php';
		require_once __DIR__ . '/core/widgets/gallery-grid-filter.php';
		require_once __DIR__ . '/core/widgets/gallery-justified.php';
		require_once __DIR__ . '/core/widgets/gallery-ken-burns.php';
		require_once __DIR__ . '/core/widgets/gallery-masonry.php';
		require_once __DIR__ . '/core/widgets/gallery-masonry-filter.php';
		require_once __DIR__ . '/core/widgets/gallery-media-grid.php';
		require_once __DIR__ . '/core/widgets/gallery-media-masonry.php';
		require_once __DIR__ . '/core/widgets/gallery-media-proofing-grid.php';
		require_once __DIR__ . '/core/widgets/gallery-media-proofing-masonry.php';
		require_once __DIR__ . '/core/widgets/gallery-packery.php';
		require_once __DIR__ . '/core/widgets/gallery-proofing-grid.php';
		require_once __DIR__ . '/core/widgets/gallery-proofing-masonry.php';
		require_once __DIR__ . '/core/widgets/gallery-ribbon.php';
		require_once __DIR__ . '/core/widgets/gallery-shift.php';
		require_once __DIR__ . '/core/widgets/gallery-slider.php';
		require_once __DIR__ . '/core/widgets/gallery-slider-content.php';
		require_once __DIR__ . '/core/widgets/gallery-split.php';
		require_once __DIR__ . '/core/widgets/before-after.php';
		require_once __DIR__ . '/core/widgets/gallery-shift-custom.php';
		require_once __DIR__ . '/core/widgets/gallery-ribbon-custom.php';

		// ------------------------ //
		// --- Register Widgets --- //
		// ------------------------ //

		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Grid_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Grid_Filtered_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Masonry_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Masonry_Filtered_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Justified_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Packery_Widget());

		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Media_Grid_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Media_Masonry_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Media_Proofing_Grid_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Media_Proofing_Masonry_Widget());

		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Proofing_Grid_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Proofing_Masonry_Widget());

		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Ken_Burns_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Ribbon_Widget());

		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Shift_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Split_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Slider_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Slider_Content_Widget());

		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Before_After_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Shift_Custom_Widget());
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type(new \Photomentor\Core\Widgets\Photomentor_Gallery_Ribbon_Custom_Widget());
	}
}

Photomentor_Plugin::instance();

if(!class_exists('Aq_Resize')) {
    class Aq_Resize
    {
        /**
         * The singleton instance
         */
        static private $instance = null;

        /**
         * No initialization allowed
         */
        private function __construct() {}

        /**
         * No cloning allowed
         */
        private function __clone() {}

        /**
         * For your custom default usage you may want to initialize an Aq_Resize object by yourself and then have own defaults
         */
        static public function getInstance() {
            if(self::$instance == null) {
                self::$instance = new self;
            }

            return self::$instance;
        }

        /**
         * Run, forest.
         */
        public function process( $url, $width = null, $height = null, $crop = null, $single = true, $upscale = false ) {
            // Validate inputs.
            if ( ! $url || ( ! $width && ! $height ) ) return false;

            // Caipt'n, ready to hook.
            if ( true === $upscale ) add_filter( 'image_resize_dimensions', array($this, 'aq_upscale'), 10, 6 );

            // Define upload path & dir.
            $upload_info = wp_upload_dir();
            $upload_dir = $upload_info['basedir'];
            $upload_url = $upload_info['baseurl'];

            $http_prefix = "http://";
            $https_prefix = "https://";

            /* if the $url scheme differs from $upload_url scheme, make them match
               if the schemes differe, images don't show up. */
            if(!strncmp($url,$https_prefix,strlen($https_prefix))){ //if url begins with https:// make $upload_url begin with https:// as well
                $upload_url = str_replace($http_prefix,$https_prefix,$upload_url);
            }
            elseif(!strncmp($url,$http_prefix,strlen($http_prefix))){ //if url begins with http:// make $upload_url begin with http:// as well
                $upload_url = str_replace($https_prefix,$http_prefix,$upload_url);
            }


            // Check if $img_url is local.
            if ( false === strpos( $url, $upload_url ) ) return false;

            // Define path of image.
            $rel_path = str_replace( $upload_url, '', $url );
            $img_path = $upload_dir . $rel_path;

            // Check if img path exists, and is an image indeed.
            if ( ! file_exists( $img_path ) or ! getimagesize( $img_path ) ) return false;

            // Get image info.
            $info = pathinfo( $img_path );
            $ext = $info['extension'];
            list( $orig_w, $orig_h ) = getimagesize( $img_path );

            // Get image size after cropping.
            $dims = image_resize_dimensions( $orig_w, $orig_h, $width, $height, $crop );
            $dst_w = $dims[4];
            $dst_h = $dims[5];

            // Return the original image only if it exactly fits the needed measures.
            if ( ! $dims && ( ( ( null === $height && $orig_w == $width ) xor ( null === $width && $orig_h == $height ) ) xor ( $height == $orig_h && $width == $orig_w ) ) ) {
                $img_url = $url;
                $dst_w = $orig_w;
                $dst_h = $orig_h;
            } else {
                // Use this to check if cropped image already exists, so we can return that instead.
                $suffix = "{$dst_w}x{$dst_h}";
                $dst_rel_path = str_replace( '.' . $ext, '', $rel_path );
                $destfilename = "{$upload_dir}{$dst_rel_path}-{$suffix}.{$ext}";

                if ( ! $dims || ( true == $crop && false == $upscale && ( $dst_w < $width || $dst_h < $height ) ) ) {
                    // Can't resize, so return false saying that the action to do could not be processed as planned.
                    return false;
                }
                // Else check if cache exists.
                elseif ( file_exists( $destfilename ) && getimagesize( $destfilename ) ) {
                    $img_url = "{$upload_url}{$dst_rel_path}-{$suffix}.{$ext}";
                }
                // Else, we resize the image and return the new resized image url.
                else {

                    $editor = wp_get_image_editor( $img_path );

                    if ( is_wp_error( $editor ) || is_wp_error( $editor->resize( $width, $height, $crop ) ) )
                        return false;

                    $resized_file = $editor->save();

                    if ( ! is_wp_error( $resized_file ) ) {
                        $resized_rel_path = str_replace( $upload_dir, '', $resized_file['path'] );
                        $img_url = $upload_url . $resized_rel_path;
                    } else {
                        return false;
                    }

                }
            }

            // Okay, leave the ship.
            if ( true === $upscale ) remove_filter( 'image_resize_dimensions', array( $this, 'aq_upscale' ) );

            // Return the output.
            if ( $single ) {
                // str return.
                $image = $img_url;
            } else {
                // array return.
                $image = array (
                    0 => $img_url,
                    1 => $dst_w,
                    2 => $dst_h
                );
            }

            return $image;
        }

        /**
         * Callback to overwrite WP computing of thumbnail measures
         */
        function aq_upscale( $default, $orig_w, $orig_h, $dest_w, $dest_h, $crop ) {
            if ( ! $crop ) return null; // Let the wordpress default function handle this.

            // Here is the point we allow to use larger image size than the original one.
            $aspect_ratio = $orig_w / $orig_h;
            $new_w = $dest_w;
            $new_h = $dest_h;

            if ( ! $new_w ) {
                $new_w = intval( $new_h * $aspect_ratio );
            }

            if ( ! $new_h ) {
                $new_h = intval( $new_w / $aspect_ratio );
            }

            $size_ratio = max( $new_w / $orig_w, $new_h / $orig_h );

            $crop_w = round( $new_w / $size_ratio );
            $crop_h = round( $new_h / $size_ratio );

            $s_x = floor( ( $orig_w - $crop_w ) / 2 );
            $s_y = floor( ( $orig_h - $crop_h ) / 2 );

            return array( 0, 0, (int) $s_x, (int) $s_y, (int) $new_w, (int) $new_h, (int) $crop_w, (int) $crop_h );
        }
    }
}

if(!function_exists('aq_resize')) {
    /**
     * This is just a tiny wrapper function for the class above so that there is no
     * need to change any code in your own WP themes. Usage is still the same :)
     */
    function aq_resize( $url, $width = null, $height = null, $crop = null, $single = true, $upscale = false ) {
        $aq_resize = Aq_Resize::getInstance();
        return $aq_resize->process( $url, $width, $height, $crop, $single, $upscale );
    }
}

# update_option('photomentor_demo', 'true');
# delete_option('photomentor_demo');
if (get_option('photomentor_demo') == 'true') {
    add_filter('body_class', function($classes) {
        return array_merge($classes, array('photomentor_demo'));
    });
}

/* Customizer Settings */
add_action('customize_register', 'photomentor_customizer_register');
if (!function_exists('photomentor_customizer_register')) {
    function photomentor_customizer_register($wp_customize)
    {
        ###################################################
        ############## Phoromentor Settings ###############
        ###################################################
        $wp_customize->add_panel('photomentor_settings',
            array(
                'title' => esc_attr__('Photomentor', 'photomentor'),
                'priority' => 200
            )
        );

        ###################################################
        ############ Lightbox Settings Section ############
        ###################################################
        $wp_customize->add_section('photomentor_lightbox_settings',
            array(
                'title' => esc_attr__('Lightbox', 'photomentor'),
                'panel' => 'photomentor_settings'
            )
        );

        # Lightbox Overlay Opacity
        $wp_setting_name = 'photomentor_lightbox_overlay';
        $wp_customize->add_setting($wp_setting_name, array('default' => '0.7', 'sanitize_callback'	=> 'esc_attr'));
        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            $wp_setting_name,
            array(
                'label' => esc_attr__('Overlay Opacity, (0 - 1)', 'photomentor'),
                'section' => 'photomentor_lightbox_settings',
                'description' => esc_html('Adjust overlay opacity from 0 to 1. (1 is 100% opacity)', 'photomentor'),
                'settings' => $wp_setting_name,
            )
        ));

        # Lightbox Title State
        $wp_setting_name = 'photomentor_lightbox_title';
        $wp_customize->add_setting($wp_setting_name, array('default' => 'none', 'sanitize_callback'	=> 'esc_attr'));
        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            $wp_setting_name,
            array(
                'label' => esc_attr__('Image Caption State', 'photomentor'),
                'section' => 'photomentor_lightbox_settings',
                'settings' => $wp_setting_name,
                'type' => 'select',
                'choices' => array('none' => 'None', 'title' => 'Use Title', 'caption' => 'Use Caption', 'description' => 'Use Description'),
            )
        ));

        # Lightbox Title Top Padding
        $wp_setting_name = 'photomentor_lightbox_padding-top';
        $wp_customize->add_setting($wp_setting_name, array('default' => '14', 'sanitize_callback'	=> 'esc_attr'));
        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            $wp_setting_name,
            array(
                'label' => esc_attr__('Caption Top Padding, PX', 'photomentor'),
                'section' => 'photomentor_lightbox_settings',
                'settings' => $wp_setting_name,
            )
        ));

        # Lightbox Title Bottom Padding
        $wp_setting_name = 'photomentor_lightbox_padding-bottom';
        $wp_customize->add_setting($wp_setting_name, array('default' => '14', 'sanitize_callback'	=> 'esc_attr'));
        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            $wp_setting_name,
            array(
                'label' => esc_attr__('Caption Bottom Padding, PX', 'photomentor'),
                'section' => 'photomentor_lightbox_settings',
                'settings' => $wp_setting_name,
            )
        ));

        # Lightbox Title Font-Size
        $wp_setting_name = 'photomentor_lightbox_font-size';
        $wp_customize->add_setting($wp_setting_name, array('default' => '16', 'sanitize_callback'	=> 'esc_attr'));
        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            $wp_setting_name,
            array(
                'label' => esc_attr__('Caption Font-Size, PX', 'photomentor'),
                'section' => 'photomentor_lightbox_settings',
                'settings' => $wp_setting_name,
            )
        ));

        # Lightbox Title Uppercase
        $wp_setting_name = 'photomentor_lightbox_uppercase';
        $wp_customize->add_setting($wp_setting_name, array('default' => false, 'sanitize_callback'	=> 'esc_attr'));
        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            $wp_setting_name,
            array(
                'label' => esc_attr__('Make Caption Uppercase', 'aurel'),
                'section' => 'photomentor_lightbox_settings',
                'settings' => $wp_setting_name,
                'type' => 'checkbox',
            )
        ));
    }
}
